/*
 * GTK See -- an image viewer based on GTK+
 * Copyright (C) 1998 Hotaru Lee <jkhotaru@mail.sti.com.cn> <hotaru@163.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * 2004-06-25: Initial release
 */
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>

#include "common_tools.h"
#include "dialog.h"
#include "gtksee.h"
#include "gtypes.h"
#include "intl.h"
#include "favorites.h"
#include "dirtree.h"
#include "rc.h"

#include "pixmaps/link_folder.xpm"

guchar            *favorites_dir;

GtkWidget         *favorites_clist;
GtkWidget         *favorites_menu;
GtkTooltips       *favorites_tooltip;

void     favorites_add           (GtkWidget *widget, gpointer data);
void     favorites_delete        (GtkWidget *widget, gpointer data);
void     favorites_rename        (GtkWidget *widget, gpointer data);
void     favorites_changeto      (GtkWidget *widget, gpointer data);
void     favorites_check_button  ();
void     favorites_path          ();
gint     motion_notify_event     (GtkWidget *widget, GdkEventMotion *event);
void     show_tips_now           (GtkWidget *w);
GtkMenu  *favorites_menu_popup   (gint row, gboolean status, GtkWidget *mytree);

/* Begin the code ... */
void
favorites_show_window(GtkWidget *fvhpaned,
                      GtkWidget *treewin,
                      GtkWidget *vpaned)
{
   static gint width_old = 3;
   static gint child_size= 2;

   gint     width, size;
   GtkPaned *winpaned;

   winpaned = GTK_PANED(fvhpaned);
   width    = GTK_WIDGET(winpaned)->allocation.width;

   if (rc_get_boolean("show_favorites"))     /* Show the Favorites window */
   {
      if (width != width_old)
      {
         child_size= child_size * width / width_old;
         width_old = width;
      }
      gtk_paned_set_position(winpaned, child_size);
      gtk_widget_show(GTK_WIDGET(vpaned));
      size = 8;
   } else                                    /* Hide the Favorites window */
   {
      if ((child_size != winpaned->child1_size) && (child_size != 2))
      {
         child_size= winpaned->child1_size + 8;
      }
      gtk_paned_set_position(winpaned, width);
      gtk_widget_hide(GTK_WIDGET(vpaned));
      size = 0;
   }

   gtk_paned_set_handle_size(winpaned, size);
   gtk_paned_set_gutter_size(winpaned, size);

   return;
}

void
favorites_add_menu(GtkWidget *clist, GtkWidget *tree)
{
   favorites_clist= clist;

   favorites_tooltip = gtk_tooltips_new();
   gtk_tooltips_set_tip(GTK_TOOLTIPS(favorites_tooltip),
       favorites_clist, "", NULL);

   gtk_signal_connect (GTK_OBJECT (favorites_clist),
      "motion_notify_event",
      GTK_SIGNAL_FUNC (motion_notify_event),
      NULL);

   gtk_signal_connect (GTK_OBJECT (favorites_clist),
      "button_press_event",
      GTK_SIGNAL_FUNC (favorites_check_button),
      tree);

   gtk_widget_set_events(favorites_clist, GDK_EXPOSURE_MASK
                         | GDK_BUTTON_PRESS_MASK
                         | GDK_LEAVE_NOTIFY_MASK
                         | GDK_POINTER_MOTION_MASK
                         | GDK_POINTER_MOTION_HINT_MASK);

}

void
favorites_check_button(GtkWidget *clist, GdkEventButton *event, gpointer data)
{
   int         row, col, result, status;
   GtkWidget   *mytree;

   switch(event->button)
   {
      case 1:  /* Left button */
         if (event->type == GDK_2BUTTON_PRESS)
         {
            result = gtk_clist_get_selection_info(GTK_CLIST(clist),
                                       event->x, event->y,
                                       &row, &col);
            if (result)
            {
               favorites_changeto(NULL, (gint *) row);
            }
         }
         break;

      case 3:  /* Right button */
         if (event->type == GDK_BUTTON_PRESS)
         {
            result = gtk_clist_get_selection_info(GTK_CLIST(clist),
                                       event->x, event->y,
                                       &row, &col);
            switch(result)
            {
               case 0:   /* Hide Change to, Browse, Delete and Rename options */
                  status = FALSE;
                  break;

               default:   /* Show Change to, Delete and Rename options */
                  status = TRUE;
                  break;
            }

            mytree = (GtkWidget *) data;
            gtk_menu_popup(favorites_menu_popup(row, status, mytree),
                              NULL, NULL, NULL, NULL,
                              event->button, event->time);
         }
         break;
   }

   return;
}

gint
favorites_load_links(GtkWidget *list, GdkWindow *gdkwin)
{
   gint           res = -1;
   gint           pos;
   gchar          *path;
   guchar         *home;
   struct stat    st;
   struct dirent  *item;
   DIR            *dp;
   gchar          *s[2];
   GdkPixmap      *pixmap;
   GdkBitmap      *mask;
   GtkStyle       *style;

   if ((home = getenv("HOME")) == NULL)
   {
      return res;
   }

   favorites_dir = gtksee_malloc(BUF_TINY);
   strncpy(favorites_dir, home, BUF_TINY);
   favorites_dir[BUF_TINY] = '\0';
   strcat(favorites_dir, "/.gtksee/Favorites/");

   if (stat(favorites_dir, &st))
   {
      g_free(favorites_dir);
      return res;
   }

   if( (dp = opendir(favorites_dir)) == NULL )
   {
      g_free(favorites_dir);
      return res;
   }

   s[0]  = NULL;
   s[1]  = gtksee_malloc(BUF_STD);

   res   = 0;
   style = gtk_widget_get_style(list);
   pixmap= gdk_pixmap_create_from_xpm_d(gdkwin,
                           &mask, &style->bg[GTK_STATE_NORMAL],
                           (gchar **)link_folder_xpm);

   while ((item = readdir(dp)) != NULL)
   {
      if (item -> d_ino == 0                  ||
            strcmp(item -> d_name, "." ) == 0 ||
            strcmp(item -> d_name, "..") == 0)
         continue;

      path = g_strdup_printf("%s%s", favorites_dir, item -> d_name);

      pos = readlink(path, s[1], BUF_STD);
      s[1][pos] = '\0';

      gtk_clist_append(GTK_CLIST(list), s);
      gtk_clist_set_pixtext( GTK_CLIST(list), res, 0,
                                   item ->d_name, 5,
                                   pixmap, mask);

      g_free(path);

      res ++;
   }

   closedir(dp);
   g_free(s[1]);

   return  res;
}

/* Internal functions */
void
favorites_add(GtkWidget *widget, gpointer data)
{
   gchar       *name;
   GdkPixmap   *pixmap;
   GdkBitmap   *mask;
   GtkStyle    *style;
   gchar       *s[2];
   gint        nrow;

   name = gtksee_malloc(BUF_STD);
   if (dialog_message(NULL, _("Favorite Folder"), _("Name"), NULL,
                        OK_CANCEL_DIALOG, FAVORITES_DIALOG,
                        TRUE, &name, BUF_TINY) == OK_STATUS)
   {
      gchar *pathname;

      style = gtk_widget_get_style(favorites_clist);
      pixmap= gdk_pixmap_create_from_xpm_d(favorites_clist->window,
                           &mask, &style->bg[GTK_STATE_NORMAL],
                           (gchar **)link_folder_xpm);

      s[0] = NULL;
      s[1] = g_strdup(dirtree_get_dir(DIRTREE(data)));

      nrow = gtk_clist_append(GTK_CLIST(favorites_clist), s);
      gtk_clist_set_pixtext  (GTK_CLIST(favorites_clist), nrow, 0,
                                      name, 5,
                                      pixmap, mask);
      pathname = g_strconcat(favorites_dir, name, NULL);

      if (symlink(s[1], pathname) == -1)
      {
         error_dialog(NULL, _("Problem creating shortcut"),
                        strerror(errno));
      }
      g_free(pathname);
      g_free(s[1]);
   }

   g_free(name);

}

void
favorites_delete(GtkWidget *widget, gpointer data)
{
   gint  row;
   gchar **name;

   row   = (gint ) data;
   name  = gtksee_malloc(BUF_STD);

   gtk_clist_get_pixtext(GTK_CLIST(favorites_clist),
                           row, 0,
                           name,
                           NULL, NULL, NULL);

   if (dialog_message(NULL, _("Favorite Folder"), *name,
                        _("Delete this shortcut ?"),
                        OK_CANCEL_DIALOG, DELETE_DIALOG,
                        FALSE, NULL, 0) == OK_STATUS)
   {
      gchar *pathname;

      pathname = g_strconcat(favorites_dir, *name, NULL);

      if (unlink(pathname) == -1)
      {
         error_dialog(NULL, _("Problem deleting shortcut"), strerror(errno));
      } else
      {
         gtk_clist_remove(GTK_CLIST(favorites_clist), row);
      }

      g_free(pathname);
   }

   g_free(name);
}

void
favorites_rename(GtkWidget *widget, gpointer data)
{
   gint        row;
   gchar       **name;
   gchar       *newname;
   GdkPixmap   *pixmap;
   GdkBitmap   *mask;
   guint8      space;


   row      = (gint ) data;
   name     = gtksee_malloc(BUF_STD);
   newname  = gtksee_malloc(BUF_STD);

   gtk_clist_get_pixtext(GTK_CLIST(favorites_clist),
                           row, 0,
                           name,
                           &space, &pixmap, &mask);

   if (dialog_message(NULL, _("Favorite Folder"), _("Rename this shortcut ?"),
                        *name,
                        OK_CANCEL_DIALOG, RENAME_DIALOG,
                        TRUE, &newname, BUF_TINY) == OK_STATUS)
   {
      gchar *pathname;
      gchar *pathnewname;

      pathname    = g_strconcat(favorites_dir, *name  , NULL);
      pathnewname = g_strconcat(favorites_dir, newname, NULL);

      if (rename(pathname, pathnewname) == -1)
      {
         error_dialog(NULL, _("Problem renaming shortcut"), strerror(errno));
      } else
      {
         gtk_clist_set_pixtext(GTK_CLIST(favorites_clist),
                                 row, 0,
                                 newname,
                                 space, pixmap, mask);
      }

      g_free(pathname);
      g_free(pathnewname);
   }

   g_free(name);
   g_free(newname);
}

void
favorites_changeto(GtkWidget *widget, gpointer data)
{
   gint        row;
   gchar       **path;


   row = (gint ) data;
   path= gtksee_malloc(BUF_STD);

   gtk_clist_get_text(GTK_CLIST(favorites_clist), row, 1, path);

   gtk_clist_select_row(GTK_CLIST(favorites_clist), row, 0);
   dir_change_enter(*path);

   g_free(path);
}

gint
motion_notify_event( GtkWidget *widget, GdkEventMotion *event)
{
   gint              pos, row, col;
   gchar             *buf = NULL;

   pos = gtk_clist_get_selection_info(GTK_CLIST(favorites_clist),
                                 event->x, event->y, &row, &col);

   if (pos)
   {
      pos = gtk_clist_get_text(GTK_CLIST(favorites_clist), row, 1, &buf);

      gtk_tooltips_set_tip(GTK_TOOLTIPS(favorites_tooltip),
            favorites_clist, buf, NULL);
      show_tips_now(favorites_clist);

      return TRUE;
   }

   return FALSE;
}

void
show_tips_now(GtkWidget *win)
{
   GdkEventCrossing  e;
   GdkWindow         *gdkwindow;

   if(GTK_WIDGET_NO_WINDOW(win))
      return;
   else
      gdkwindow = win->window;

   if(gdkwindow == NULL)
      return;

/* Send a fake enter notify event to make widget think
 * its time to show the tooltips. Note that the widget
 * should be watching for the enter_notify_event.
 */
   e.type      = GDK_ENTER_NOTIFY;
   e.window    = gdkwindow;
   e.send_event= 1;               /* True if we're sending event. */
   e.subwindow = gdkwindow;
   e.time      = GDK_CURRENT_TIME;
   e.x         = 610.0;
   e.y         = 610.0;
   e.x_root    = 520.0;
   e.y_root    = 520.0;
   e.mode      = GDK_CROSSING_NORMAL;
   e.detail    = GDK_NOTIFY_ANCESTOR;
   e.focus     = TRUE;                 /* Focus. */
   e.state     = 0;                    /* Key modifiers. */
   gdk_event_put((GdkEvent *)&e);
}

GtkMenu *
favorites_menu_popup(gint row, gboolean status, GtkWidget *mytree)
{
   GtkWidget *menu = NULL;
   GtkWidget *menu_item;

   menu = gtk_menu_new();

   menu_item = gtk_menu_item_new_with_label(_("Change to..."));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(favorites_changeto), (gint *)row);
   gtk_widget_set_sensitive(menu_item, status);

   menu_item = gtk_menu_item_new();
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_widget_show(menu_item);

   menu_item = gtk_menu_item_new_with_label(_("Add"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(favorites_add), mytree);

   menu_item = gtk_menu_item_new_with_label(_("Delete"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(favorites_delete), (gint *)row);
   gtk_widget_set_sensitive(menu_item, status);

   menu_item = gtk_menu_item_new_with_label(_("Rename"));
   gtk_menu_append(GTK_MENU(menu), menu_item);
   gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
      GTK_SIGNAL_FUNC(favorites_rename), (gint *)row);
   gtk_widget_set_sensitive(menu_item, status);

   gtk_widget_show_all(menu);

   return GTK_MENU(menu);
}
