# -*- coding: utf-8 -*-

import gobject
import gtk
from workview import WorkView
from openxml import OpenXML
from savexml import SaveXML
from configfile import ConfigFile
from importutf import ImportUtf
from importlist import ImportList
from net import StrokeClient


_WINDOW_UI = \
"""
<ui>
  <menubar name="MenuBar">
    <menu action="FileMenu">
      <menuitem action="Open"/>
      <menuitem action="Save"/>
      <menuitem action="ImportUtf"/>
      <menuitem action="ImportList"/>
      <menuitem action="ImportToServer"/>
      <menuitem action="Connect"/>
      <separator/>
      <menuitem action="Quit"/>
    </menu>
    <menu action="EditMenu">
      <menuitem action="Add"/>
      <menuitem action="DeleteKey"/>
      <menuitem action="EditKey"/>
      <menuitem action="Search"/>
      <separator/>
      <menu action="StrokeMenu">
        <menuitem action="Cut"/>
        <menuitem action="Copy"/>
        <menuitem action="Paste"/>
        <menuitem action="Delete"/>
        <separator/>
        <menuitem action="SelectAll"/>
      </menu>
      <separator/>
      <menuitem action="Font"/>
    </menu>
    <menu action="HelpMenu">
      <menuitem action="Help"/>
      <menuitem action="About"/>
    </menu>
  </menubar>
   
  <toolbar name="Toolbar">
    <toolitem action="Open"/>
    <toolitem action="Save"/>
    <toolitem action="Connect"/>
    <separator/>
    <toolitem action="Add"/>
    <toolitem action="DeleteKey"/>
    <toolitem action="EditKey"/>
    <toolitem action="Search"/>
    <separator/>
    <toolitem action="About"/>
  </toolbar>
</ui>
"""

class MainWindow(gtk.Window):
	def __init__(self):
		self.config = ConfigFile()
		gtk.Window.__init__(self)
		self.set_title("Stroke Editor")
		self.connect ("destroy", self.do_quit)
		vbox = gtk.VBox(False, 2)
		self.add (vbox)
		# Actions entries
		self._entries = (
			( "FileMenu", None, "_File" ),
			( "EditMenu", None, "_Edit" ),
			( "HelpMenu", None, "_Help" ),
			( "StrokeMenu", None, "_Stroke" ),
			# (action name, stock id, label, accelerator, tooltip, callback)
			( "Open", gtk.STOCK_OPEN,
				None, None, None, self.do_open),
			( "ImportUtf", gtk.STOCK_CONVERT,
				"Import utf file", None, None, self.do_importUtf),
			( "ImportList", gtk.STOCK_CONVERT,
				"Import list file", None, None, self.do_importList),
			( "ImportToServer", gtk.STOCK_CONVERT,
				"Import To Server", None, None, self.do_import_to_server),
			( "Save", gtk.STOCK_SAVE,
				None, None, None, self.do_save),
			( "Connect", gtk.STOCK_CONNECT,
				None, None, None, self.do_connect),
			( "Cut", gtk.STOCK_CUT,
				None, None, None, self.do_cut),
			( "Copy", gtk.STOCK_COPY,
				None, None, None, self.do_copy),
			( "Paste", gtk.STOCK_PASTE,
				None, None, None, self.do_paste),
			( "Delete", gtk.STOCK_DELETE,
				None, None, None, self.do_delete),
#			( "SelectAll", gtk.STOCK_SELECT_ALL,
#				None, "<Ctrl>A", None, self.do_select_all),
			( "SelectAll", None,
				"Select _All", "<Ctrl>A", None, self.do_select_all),
			( "Add", gtk.STOCK_ADD,
				None, None, None, self.do_newkey),
			( "Search", gtk.STOCK_FIND,
				None, None, None, self.do_search),
			( "DeleteKey", gtk.STOCK_DELETE,
				"Delete Key", None, None, self.do_delkey),
			( "EditKey", gtk.STOCK_EDIT,
				"Edit Key", None, None, self.do_editkey),
			( "Font", gtk.STOCK_SELECT_FONT,
				None, None, None, self.do_font),
			( "Quit", gtk.STOCK_QUIT,
				None, None, None, self.do_quit),
			( "Help", gtk.STOCK_HELP,
				None, None, None, None),
			( "About", gtk.STOCK_ABOUT,
				None, None, None, self.do_about),
		)
		
		actions = gtk.ActionGroup ("Actions")
		actions.add_actions (self._entries)
			
		self._manager = gtk.UIManager ()
		self._manager.insert_action_group (actions, 1)
		self.add_accel_group (self._manager.get_accel_group ())
		self._manager.add_ui_from_string (_WINDOW_UI)

		self._manager.get_widget("/Toolbar/DeleteKey").set_homogeneous(False)
		self._manager.get_widget("/Toolbar/EditKey").set_homogeneous(False)
		vbox.pack_start(self._manager.get_widget ("/MenuBar"), False)
		vbox.pack_start(self._manager.get_widget ("/Toolbar"), False)

		hbox = gtk.HBox(False, 5)
		vbox.pack_start(hbox, True, True)
		hbox.pack_start(self.create_charlist(), False)

		self.workview = WorkView(self.config)
		self.workview.connect("strokes-changed", self.on_workview_strokes_changed)
		self.workview.get_selection().connect ("changed", self.on_workview_selection_changed)
		hbox.pack_start(self.workview, True, True)
		self.statusbar = gtk.Statusbar()
		vbox.pack_start(self.statusbar, False)
		self._manager.get_action("/MenuBar/EditMenu/StrokeMenu").set_sensitive(False)


		self.update_statusbar()
		self.enable_network = False
		self._current_chars = False
		self._unit = int(self.config.get("server", "unit", "0"))
		self._url = self.config.get("server", "url", "http://10.66.0.143:1980")

	def create_charlist(self):
		self._manager.get_action("/Toolbar/DeleteKey").set_sensitive(False)				
		self._manager.get_action("/Toolbar/EditKey").set_sensitive(False)				
		self.char_liststore = gtk.ListStore(gobject.TYPE_STRING, object, gobject.TYPE_BOOLEAN)
		self.char_treeview = gtk.TreeView(self.char_liststore)
		charcolumn = gtk.TreeViewColumn("Key")
		self.char_treeview.append_column(charcolumn)
		cell = gtk.CellRendererText()
		charcolumn.pack_start(cell, True)
		charcolumn.add_attribute(cell, 'text', 0)
		charcolumn.set_sort_column_id(0)
		strokescolumn  = gtk.TreeViewColumn("Done")
		self.char_treeview.append_column(strokescolumn)
		cell = gtk.CellRendererToggle()
		strokescolumn.pack_start(cell, True)
		strokescolumn.add_attribute(cell, 'active', 2)
		strokescolumn.set_sort_column_id(2)
		treeselection = self.char_treeview.get_selection()
		treeselection.connect("changed", self.on_char_treeview_selected)
		scrolled_window = gtk.ScrolledWindow()
		scrolled_window.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
		scrolled_window.add(self.char_treeview)
		scrolled_window.set_size_request(115, -1)
		return scrolled_window

	def on_char_treeview_selected(self, treeselection):
		(model, iter) = treeselection.get_selected()
		self._manager.get_action("/Toolbar/DeleteKey").set_sensitive(bool(iter))
		self._manager.get_action("/Toolbar/EditKey").set_sensitive(bool(iter))
		if iter:			
			if self.enable_network:
				if self._current_chars:
					strokes = self.workview.get_strokes()
					if self.workview.is_modified ():
						self.client.update_strokes (self._current_chars, strokes)
				chars = unicode (model.get_value(iter, 0), "UTF-8")
				strokes = self.client.get_strokes (chars)
				self.workview.set_chars (chars, strokes)
				self._current_chars = chars
			else:
				self.workview.set_chars(unicode(model.get_value(iter, 0), "UTF-8"), model.get_value(iter, 1))


	def on_workview_strokes_changed(self, workview):
		treeselection = self.char_treeview.get_selection()
		(model, iter) = treeselection.get_selected()
		if iter:
			strokes = workview.get_strokes()
			model.set_value(iter, 1, strokes)
			b = bool(strokes)
			if model.get_value (iter, 2) != b:
				model.set_value(iter, 2, bool(strokes))
				self.update_statusbar ()

	def on_workview_selection_changed(self, selection):
		if selection:
			list, rows = selection.get_selected_rows ()
			sel = (len(rows) >= 1)
			self._manager.get_action("/MenuBar/EditMenu/StrokeMenu").set_sensitive(sel)

	def do_connect(self, widget):
		dlg = gtk.Dialog ("Select Unit", 
			buttons = (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
				gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
		entry = gtk.Entry ()
		entry.show ()
		entry.set_text (self._url)
		dlg.vbox.add (entry)
		
		spin = gtk.SpinButton( gtk.Adjustment(self._unit,-1,1000,1), digits=0)
		spin.show()
		dlg.vbox.add (spin)
		if dlg.run() != gtk.RESPONSE_ACCEPT:
			dlg.destroy ()
			return
		self._unit = int(spin.get_value() )
		self._url = entry.get_text()
		self.client = StrokeClient (self._url)
		self.enable_network = True
		self.config.set ("server", "url", self._url)
		self.config.set ("server", "unit", str(self._unit))
		self.config.save ()
		if self._unit == -1:
			all = self.client.get_all_chars ()
		else:
			all = self.client.get_chars_by_range (int( self._unit * 100), 100)
		self.char_liststore = gtk.ListStore(gobject.TYPE_STRING, object, gobject.TYPE_BOOLEAN)
		for c in all: 
			self.char_liststore.append ([c[0].encode("UTF-8"), [], bool(c[1])])
		self.char_treeview.set_model(self.char_liststore)
		self.update_statusbar()
		dlg.destroy()
		
	def do_importUtf(self, widget):
		self.char_treeview.set_model(None)
		importutf = ImportUtf(self)
		importutf.do_import()
		self.char_treeview.set_model(self.char_liststore)
		self.update_statusbar()

	def do_importList(self, widget):
		self.char_treeview.set_model(None)
		importlist = ImportList(self)
		importlist.do_import()
		self.char_treeview.set_model(self.char_liststore)
		self.update_statusbar()

	def do_import_to_server(self, widget):
		dlg = gtk.Dialog ("Select Unit", 
			buttons = (gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT,
				gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
		entry = gtk.Entry ()
		entry.show ()
		entry.set_text (self._url)
		dlg.vbox.add (entry)
		
		if dlg.run() != gtk.RESPONSE_ACCEPT:
			dlg.destroy ()
			return
		self.client = StrokeClient (self._url)
		model = self.char_treeview.get_model()
		for i in model:
			if i[2]:
				self.client.update_strokes(unicode(i[0],"utf8"), i[1])
		dlg.destroy()
		
	def do_newkey(self, widget):
		dialog = NewKeyDialog(self, self.char_treeview, self.char_liststore)
		dialog.do_newkey()
		self.update_statusbar()

	def do_delkey(self, widget):
		treeselection = self.char_treeview.get_selection()
		(model, iter) = treeselection.get_selected()
		if iter:
			dialog = gtk.MessageDialog(None, gtk.DIALOG_MODAL, gtk.MESSAGE_QUESTION, gtk.BUTTONS_YES_NO, "Are you sure?")
			retvar =dialog.run()
			if retvar == gtk.RESPONSE_YES:
				model.remove(iter)
			dialog.destroy()
			self.update_statusbar()

	def do_editkey(self, widget):
		treeselection = self.char_treeview.get_selection()
		(model, iter) = treeselection.get_selected()
		if iter:
			dialog = EditKeyDialog(self, self.char_treeview)
			dialog.do_editkey()

	def do_font (self, widget):
		dialog = gtk.FontSelectionDialog ("Font")
		dialog.set_font_name (self.workview.get_font ())
		retval = dialog.run()

		if retval == gtk.RESPONSE_OK:
			char_font = dialog.get_font_name()
			self.workview.set_font (char_font)
			self.config.set("font", "char_font", char_font)
		dialog.destroy()

	def do_search(self, widget):
		searchdlg = SearchDialog(self, self.char_treeview, self.char_liststore)
		searchdlg.do_search()

	def do_quit(self, widget):
		self.config.save()
		gtk.main_quit()

	def do_open(self, widget):
		self.char_treeview.set_model(None)
		openxml = OpenXML(self)
		openxml.do_open()
		self.char_treeview.set_model(self.char_liststore)
		self.update_statusbar()

	def do_copy (self, widget):
		self.workview.copy_selected_strokes ()

	def do_cut (self, widget):
		self.workview.copy_selected_strokes ()
		self.workview.delete_selected_strokes()

	def do_paste (self, widget):
		self.workview.paste_strokes()

	def do_delete (self, widget):
		self.workview.delete_selected_strokes()

	def do_select_all (self, widget):
		self.workview.select_all ()

	def do_save (self, widget):
		if self.enable_network:
			for i in self.char_liststore:
				if i[2] and (bool(i[1]) == False):
					chars = unicode (i[0], "UTF-8")
					strokes = self.client.get_strokes (chars)
					i[1] = strokes
		savexml = SaveXML(self.char_liststore)
		savexml.do_save()

	def do_about(self, widget):
		about = gtk.AboutDialog()
		about.set_name("Stroke Editor")
		about.set_version("1.3")
		about.set_copyright("Copyright \xc2\xa9 2007 by RedHat Inc.")
		about.set_comments("Stroke Editor is a tool for the Tomoe project.")
		about.set_website("http://tomoe.sourceforge.net")
		about.set_authors(["Hu Zheng <zhu@redhat.com>", "Huang Peng <phuang@redhat.com>"])
		about.set_transient_for (self)
		about.set_position (gtk.WIN_POS_CENTER_ON_PARENT)
		about.run()
		about.destroy()

	def update_statusbar(self):
		total = self.char_liststore.iter_n_children(None)
		finished = 0
		empty = 0
		for i in self.char_liststore:
			if i[2]:
				finished = finished + 1
			else:
				empty = empty + 1
		info = str(total) + " characters: " + str(finished) + " finished, " + str(empty) + " empty."
		self.statusbar.pop(0)
		self.statusbar.push(0, info)

class NewKeyDialog(gtk.Dialog):
	def __init__(self, parent, treeview, liststore):
		gtk.Dialog.__init__(self, "Input the new character", parent, gtk.DIALOG_MODAL, (gtk.STOCK_OK, gtk.RESPONSE_OK))
		self.set_default_response (gtk.RESPONSE_OK)
		self.set_position (gtk.WIN_POS_CENTER_ON_PARENT)
		self.treeview = treeview
		self.liststore = liststore
		hbox = gtk.HBox(False, 5)
		hbox.show()
		label = gtk.Label("New key:")
		label.show()
		hbox.pack_start(label, False)
		self.entry = gtk.Entry()
		self.entry.connect("activate", self.on_entry_activate)
		self.entry.show()
		hbox.pack_start(self.entry, True)
		self.vbox.pack_start(hbox, True, True, 5)

	def on_entry_activate(self, widget):
		self.response(gtk.RESPONSE_OK)

	def do_newkey(self):
		retval = self.run()
		if retval == gtk.RESPONSE_OK:
			utf8char = self.entry.get_text()
			if utf8char != "":
				treeiter = self.liststore.get_iter_first()
				while treeiter:
					utf8char_tmp = self.liststore.get_value(treeiter, 0)
					if utf8char_tmp == utf8char:
						break
					treeiter = self.liststore.iter_next(treeiter)
				if treeiter == None:
					treeiter = self.liststore.append([utf8char, [], False])
				self.treeview.get_selection().select_iter(treeiter)
				self.treeview.scroll_to_cell(self.liststore.get_path(treeiter))
		self.destroy()

class EditKeyDialog(gtk.Dialog):
	def __init__(self, parent, treeview):
		gtk.Dialog.__init__(self, "Edit the key", parent, gtk.DIALOG_MODAL, (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
		self.set_default_response (gtk.RESPONSE_OK)
		self.set_position (gtk.WIN_POS_CENTER_ON_PARENT)
		self.parent_window = parent
		self.treeview = treeview
		hbox = gtk.HBox(False, 5)
		hbox.show()
		label = gtk.Label("Edit key:")
		label.show()
		hbox.pack_start(label, False)
		self.entry = gtk.Entry()
		treeselection = treeview.get_selection()
		(model, iter) = treeselection.get_selected()
		if iter:
			utf8str = model.get_value (iter, 0)
			self.entry.set_text(utf8str)
			self.entry.select_region(0, -1)
		self.entry.connect("activate", self.on_entry_activate)
		self.entry.show()
		hbox.pack_start(self.entry, True)
		self.vbox.pack_start(hbox, True, True, 5)

	def on_entry_activate(self, widget):
		self.response(gtk.RESPONSE_OK)

	def do_editkey(self):
		retval = self.run()
		if retval == gtk.RESPONSE_OK:
			utf8str = self.entry.get_text()
			if utf8str != "":
				treeselection = self.treeview.get_selection()
				(model, iter) = treeselection.get_selected()
				if iter:
					model.set_value(iter, 0, utf8str)
					self.parent_window.on_char_treeview_selected(treeselection)
		self.destroy()

class SearchDialog(gtk.Dialog):
	def __init__(self, parent, treeview, liststore):
		gtk.Dialog.__init__(self, "Input the search character", parent, gtk.DIALOG_MODAL, (gtk.STOCK_OK, gtk.RESPONSE_OK))
		self.set_default_response (gtk.RESPONSE_OK)
		self.set_position (gtk.WIN_POS_CENTER_ON_PARENT)
		self.treeview = treeview
		self.liststore = liststore
		hbox = gtk.HBox(False, 5)
		hbox.show()
		label = gtk.Label("Search character:")
		label.show()
		hbox.pack_start(label, False)
		self.entry = gtk.Entry()
		self.entry.connect("activate", self.on_entry_activate)
		self.entry.show()
		hbox.pack_start(self.entry, True)
		self.vbox.pack_start(hbox, True, True, 5)
		label = gtk.Label("Leave entry empty to find the first empty strokes char.")
		label.show()
		self.vbox.pack_start(label, False, False, 5)

	def on_entry_activate(self, widget):
		self.response(gtk.RESPONSE_OK)

	def do_search(self):
		retval = self.run()
		if retval == gtk.RESPONSE_OK:
			utf8char = self.entry.get_text()
			if utf8char == "":
				treeiter = self.liststore.get_iter_first()
				while treeiter:
					chardone = self.liststore.get_value(treeiter, 2)
					if chardone == False:
						break
					treeiter = self.liststore.iter_next(treeiter)
				if treeiter == None:
					dialog = gtk.MessageDialog(None, gtk.DIALOG_MODAL, gtk.MESSAGE_INFO, gtk.BUTTONS_CLOSE, "No empty strokes char found!")
					dialog.run()
					dialog.destroy()
				else:
					self.treeview.get_selection().select_iter(treeiter)
					self.treeview.scroll_to_cell(self.liststore.get_path(treeiter))
			else:
				treeiter = self.liststore.get_iter_first()
				while treeiter:
					utf8char_tmp = self.liststore.get_value(treeiter, 0)
					if utf8char_tmp == utf8char:
						break
					treeiter = self.liststore.iter_next(treeiter)
				if treeiter == None:
					dialog = gtk.MessageDialog(None, gtk.DIALOG_MODAL, gtk.MESSAGE_INFO, gtk.BUTTONS_CLOSE, "Not found!")
					dialog.run()
					dialog.destroy()
				else:
					self.treeview.get_selection().select_iter(treeiter)
					self.treeview.scroll_to_cell(self.liststore.get_path(treeiter))
		self.destroy()

def main():
	w = MainWindow()
	w.show_all()
	gtk.main()

if __name__ == "__main__":
	main()
