/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   copyright            : (C) 2003 by Zhang Yong                         *
 *   email                : z-yong163@163.com                              *
 ***************************************************************************/

#ifndef _UDP_SESSION_H
#define _UDP_SESSION_H

#include "udppacket.h"
#include "socket.h"
#include <list>

using namespace std;

#define MAX_SEND_ATTEMPTS	3
#define SEND_TIMEOUT		10
#define KEEPALIVE_TIMEOUT	120


enum {
	CMD_ACK = 1,
	CMD_KEEPALIVE,
	CMD_REGISTER,
	CMD_PRE_LOGIN,
	CMD_LOGIN,
	CMD_LOGOUT,

	CMD_CHANGE_STATUS = 0x1000,
	CMD_ADD_CONTACT,
	CMD_DEL_CONTACT,
	CMD_MESSAGE,
	CMD_SEARCH,
	CMD_SEARCH_RANDOM,
	CMD_GET_CONTACT_INFO,
	CMD_GET_USER_INFO,
	CMD_UPDATE_USER_INFO,
	CMD_GET_CONTACT_LIST,
	CMD_PRE_CHANGE_PASSWD,
	CMD_CHANGE_PASSWD,

	CMD_SRV_USER_ONLINE = 0x2000,
	CMD_SRV_USER_OFFLINE,
	CMD_SRV_USER_STATUS,
	CMD_SRV_PROBE_STATUS,
	CMD_SRV_SEARCH_RESULT,
	CMD_SRV_ADD_CONTACT,
	CMD_SRV_DEL_CONTACT,
	CMD_SRV_CONTACT_AUTH,
	CMD_SRV_MESSAGE,
};


enum {
	LOGIN_SUCCESS,
	LOGIN_INVALID_USER,
	LOGIN_WRONG_PASSWD,
	LOGIN_ERROR_UNKNOWN,
};

enum {
	REG_SUCCESS,
	REG_USER_EXISTS,
	REG_ERROR_UNKNOWN,
};

enum {
	AUTH_ACCEPTED,
	AUTH_REQUEST,
	AUTH_REJECTED,
};

enum {
	STATUS_OFFLINE,
	STATUS_ONLINE,
	STATUS_AWAY,
	STATUS_INVIS
};

enum {
	MSG_TEXT,
	MSG_AUTH_ACCEPTED,
	MSG_AUTH_REQUEST,
	MSG_AUTH_REJECTED,
	MSG_ADDED,
	MSG_TCP_REQUEST,
	MSG_TCP_ACCEPT,
};


struct ONLINE_INFO {
	const char *name;
	uint32 status;
	uint16 tcp_ver;
	uint32 ip;
	uint32 real_ip;
	uint16 msg_port;
};

enum {
	GENDER_UNSPECIFIED,
	GENDER_MALE,
	GENDER_FEMALE,
};

struct SEARCH_RESULT {
	const char *name;
	uint32 status;
	const char *nick;
	uint8 auth;
	uint8 gender;
	uint8 age;
};

struct CONTACT_INFO {
	const char *name;
	const char *nick;
	uint8 gender;
	uint32 birth;
	const char *email;
	const char *country;
	const char *city;
	const char *address;
	const char *postcode;
	const char *tel;
	const char *mobile;
	const char *realname;
	const char *occupation;
	const char *homepage;
	const char *intro;
};

struct USER_INFO : public CONTACT_INFO {
	uint8 auth;
};


class UDPSessionListener {
public:
	virtual void onAck(uint32 seq) = 0; // yes, uint32, not uint16, this fit p2p/MessageSession::onMessageAck() too. but, need to move to class ICQMain?
	virtual void onSendError(uint16 seq) = 0;
	virtual void onConnect(bool connected) = 0;

	virtual void onRegisterReply(uint8 error) = 0;
	virtual void onPreLoginReply(uint16 sequence, const char *token) = 0;
	virtual void onLoginReply(uint8 error) = 0;
	virtual void onAddContactReply(const char *name, uint8 auth) = 0;
	virtual void onContactListReply(const char *list[], int n) = 0;

	virtual void onUserOnline(ONLINE_INFO &info) = 0;
	virtual void onUserOffline(const char *name) = 0;
	virtual void onUserStatus(const char *name, uint32 status) = 0;
	virtual void onSearchResult(SEARCH_RESULT result[], int n) = 0;
	virtual void onRecvMessage(uint8 type, const char *from, time_t when, const char *text) = 0;

	virtual void onContactInfoReply(CONTACT_INFO &c) = 0;
	virtual void onUserInfoReply(USER_INFO &user) = 0;
	virtual void onPreChangePasswordReply(uint16 sequence, const char *token) = 0;
};


class UDPSession : public SocketListener {
public:
	UDPSession(UDPSessionListener *l);
	virtual ~UDPSession();

	virtual void onSocketRead();
	virtual void onSocketWrite();
	virtual void onSocketException();

	uint16 getLastSeq() { return sendSeq; }
	bool isBehindWall() { return realIP != ourIP; }

	void connect(Socket *sock, const char *host, int port);
	void checkSendQueue();
	void sendKeepAlive();
	void registerUser(const char *name, const char *passwd);
	void preLogin(const char *name);
	void login(const char *name, const char *passwd, const char *token, uint16 sequence, uint32 status, uint16 port);
	void logout();
	void changeStatus(uint32 status);
	void sendMessage(uint8 type, const char *to, const char *text);
	void searchRandom();
	void searchUser(const char *name, const char *nick, const char *email);
	void addContact(const char *name);
	void delContact(const char *name);
	void getContactList();
	void getContactInfo(const char *name);
	void getUserInfo();
	void updateUserInfo(USER_INFO &info);
	void preChangePassword();
	void changePassword(const char *oldPasswd, const char *token, uint16 sequence, const char *newPasswd);

	void onPacketReceived(UDPInPacket &in);

	uint32 numClients;

private:
	void reset();
	void clearSendQueue();
	bool checkSeq(uint16 seq);

	void createPacket(UDPOutPacket &out, uint16 cmd, uint16 seq);
	UDPOutPacket *createPacket(uint16 cmd);
	void sendAck(uint16 seq);
	void sendPacket(UDPOutPacket *out);
	void sendDirect(UDPOutPacket *out);

	void onAck(UDPInPacket &in);
	void onRegisterReply(UDPInPacket &in);
	void onPreLoginReply(UDPInPacket &in);
	void onLoginReply(UDPInPacket &in);
	void onKeepAliveReply(UDPInPacket &in);
	void onAddContactReply(UDPInPacket &in);
	void onContactListReply(UDPInPacket &in);

	void onUserOnline(UDPInPacket &in);
	void onUserOffline(UDPInPacket &in);
	void onUserStatus(UDPInPacket &in);
	void onSearchResult(UDPInPacket &in);
	void onRecvMessage(UDPInPacket &in);
	void onContactInfoReply(UDPInPacket &in);
	void onUserInfoReply(UDPInPacket &in);
	void onPreChangePasswordReply(UDPInPacket &in);

	UDPSessionListener *listener;
	Socket *udpSocket;

	uint32 sessionID;
	uint32 realIP, ourIP;
	uint16 sendSeq, recvSeq;
	uint8 seqWindow[1 << 13];
	list<UDPOutPacket *> sendQueue;
};


#endif
