/*
	writer	: Ma Su An
	E-Mail	: msa@wri.com.cn
	
	Copyright by Ma Su An.
	All rights reserved.
	Permission to use ,copy,modify,and distribute this software for
	individual use and without fee is granted with that condition:

    	Every copy of this software must have the writer's name displayed 
	on the top label.
*/

// filename: image.cc
// msa 1999.1

#include "image.h"
#include <assert.h>

/****************************************************************************/
Gif::Gif( char *filename )
{
    if ( filename != NULL && strlen( filename ) < LINELENGTH - 1 )
    {
        strcpy( gifname, filename );
    } else
    {
        gifname[0] = '\0';
    }
    return;
}

/****************************************************************************/
void
Gif::vSetFileName( char *filename )
{
    if ( filename != NULL && strlen( filename ) < LINELENGTH - 1 )
    {
        strcpy( gifname, filename );
    }

    return;
}

/****************************************************************************/
int
Gif::iGetGif( ImageInfo * imageinfo )
{
    FILE *giffile;
    int err = 0;
    int imagesize;

    if ( !( giffile = fopen( gifname, "rb" ) ) )
    {
        return ( errno );
    }

    datapoint = 0;
    index = 0;
    remcnt = 0;
    err = gif_header( &info, giffile );
    if ( err )
        return ( err );

    imagesize = info.width * info.height;
    if ( !( info.data = ( char * ) malloc( imagesize ) ) )
    {
        return ( errno );
    }

    err = unpack_gif( &info, giffile );
    fclose( giffile );

    if ( !err )
    {
        adjust( &info );
        countcolor( &info );
    }

    *imageinfo = info;

    return ( err );
}

/***********************************************************************/
int
Gif::unpack_gif( ImageInfo * p_gifinfo, FILE * giffile )
{
    short int code, oldcode, first_ch;
    short int CLEAR, EOI;

    fseek( giffile, headsize, 0 );      /*jump over the gif header. */
    first_ch = getc( giffile );
    CLEAR = 1 << first_ch;
    EOI = CLEAR + 1;

    reqcnt = first_ch + 1;      /*reqcnt means how many bits a code want. */

    oldcode = -1;
    nextcode = CLEAR + 2;

    CTlink = ( short int * ) malloc( LISTLENGTH * 2 );
    CTfirst = ( char * ) malloc( LISTLENGTH );
    CTlast = ( char * ) malloc( LISTLENGTH );
    OStack = ( char * ) malloc( LISTLENGTH );

    if ( !CTlink || !CTfirst || !CTlast || !OStack )
    {
        return ( errno );
    }

    while ( ( code = get_code( reqcnt, giffile ) ) != EOI && !feof( giffile ) )
    {
        if ( code == CLEAR )
        {
            unpack_init( CLEAR );
            oldcode = -1;
            nextcode = CLEAR + 2;
            reqcnt = first_ch + 1;
            nextlim = 1 << reqcnt;
        } else
        {
            if ( CTlink[code] == -2 )
            {
                insert_ct( oldcode, oldcode );
            } else
            {
                if ( oldcode != -1 )
                {
                    insert_ct( code, oldcode );
                }
            }
            putx( code, p_gifinfo->data );
            oldcode = code;
        }
    }

    free( ( char * ) CTlink );
    free( CTlast );
    free( CTfirst );
    free( OStack );

    if ( code != EOI )
        return ( EOIERR );

    return ( 0 );
}

/***********************************************************************/
int
Gif::gif_header( ImageInfo * gifinfo, FILE * giffile )
{
    GLOBALHEADER globalh;
    LOCALHEADER localh;
    short int plsize, bitsofpixel;

    rewind( giffile );
    headsize = 13;
    fread( ( char * ) globalh.version, 6, 1, giffile );

    if ( !strncmp( globalh.version, "gif", 3 ) )
    {
        return ( NOTGIF );
    }

    fread( ( char * ) &globalh.screen_width, 2, 1, giffile );
    fread( ( char * ) &globalh.screen_height, 2, 1, giffile );
    globalh.global_flag = getc( giffile );
    globalh.bg_color = getc( giffile );
    globalh.zero = getc( giffile );

    gifinfo->bg_color = globalh.bg_color;

    exchange( &( globalh.screen_width ) );
    exchange( &( globalh.screen_height ) );

    if ( globalh.global_flag & 0x80 )
    {
        bitsofpixel = ( globalh.global_flag & 0x7 ) + 1;
        plsize = 1 << bitsofpixel;
        gifinfo->colornum = plsize;
        headsize += plsize * 3;
        fread( ( char * ) ( gifinfo->palette ), plsize * 3, 1, giffile );
    }

    headsize += 10;

    localh.comma = getc( giffile );
    fread( ( char * ) &localh.image_left, 2, 1, giffile );
    fread( ( char * ) &localh.image_top, 2, 1, giffile );
    fread( ( char * ) &localh.image_width, 2, 1, giffile );
    fread( ( char * ) &localh.image_height, 2, 1, giffile );
    localh.local_flag = getc( giffile );

    if ( localh.comma != ',' )
    {
        return ( SPECIALFILE );
    }

    exchange( &( localh.image_width ) );
    exchange( &( localh.image_height ) );
    exchange( &( localh.image_left ) );
    exchange( &( localh.image_top ) );

    if ( localh.local_flag & 0x80 )
    {
        bitsofpixel = ( localh.local_flag & 0x7 ) + 1;
        plsize = 1 << bitsofpixel;
        gifinfo->colornum = plsize;
        headsize += plsize * 3;
        fread( ( char * ) ( gifinfo->palette ), plsize * 3, 1, giffile );
    }

    gifinfo->width = localh.image_width;
    gifinfo->height = localh.image_height;
    interlaced = localh.local_flag & 0x40;

    return ( 0 );
}

/***********************************************************************/
void
Gif::countcolor( ImageInfo * gifinfo )
{
    int i, size, maxcolor = 0;

    size = gifinfo->width * gifinfo->height;
    for ( i = 0; i < size; ++i )
    {
        if ( ( unsigned char ) gifinfo->data[i] > maxcolor )
            maxcolor = ( unsigned char ) gifinfo->data[i];
    }

    gifinfo->colornum = maxcolor + 1;

    return;
}

/*********************************************************************/
void
Gif::adjust( ImageInfo * gifinfo )
{
    short int *linelist;
    char *buff, *p, *q;
    int i, j, notfinished, scan1end, line_in_buff, scan2end, scan3end;
    short int height, width;

    if ( !interlaced )
        return;

    height = gifinfo->height;
    width = gifinfo->width;

    linelist = ( short int * ) malloc( ( height + 1 ) * sizeof( short int ) );
    buff = ( char * ) malloc( width );

    scan1end = height / 8;
    if ( height % 8 >= 1 )
        scan1end++;

    scan2end = scan1end;
    scan2end += height / 8;
    if ( height % 8 >= 5 )
        scan2end++;

    scan3end = scan2end;
    scan3end += height / 4;
    if ( height % 4 >= 3 )
        scan3end++;

    for ( i = 0; i < height; ++i )
    {
        if ( i % 8 == 0 )
            linelist[i] = i / 8;
        else if ( i % 4 == 0 )
            linelist[i] = i / 8 + scan1end;
        else if ( i % 2 == 0 )
            linelist[i] = i / 4 + scan2end;
        else
            linelist[i] = i / 2 + scan3end;
    }

    notfinished = 1;
    while ( notfinished )
    {
        for ( i = 0; linelist[i] == i && i < height; i++ );

        if ( i == height )
            notfinished = 0;
        else
        {
            p = gifinfo->data + i * width;
            memcpy( buff, p, width );
            line_in_buff = i;
            while ( linelist[i] != line_in_buff )
            {
                q = gifinfo->data + i * width;
                p = gifinfo->data + linelist[i] * width;
                memcpy( q, p, width );
                j = i;
                i = linelist[i];
                linelist[j] = j;
            }
            q = gifinfo->data + i * width;
            memcpy( q, buff, width );
            linelist[i] = i;
        }
    }
    free( buff );
    free( linelist );

    return;
}

/*******************************************************/
void
Gif::exchange( unsigned short int *numberp )
{
/*  used in sun sparc workstation:

        unsigned short int low,high;
 
        low=(*numberp)>>8;
        high=(*numberp)<<8;
        low&=0x00ff;
        high&=0xff00;
        *numberp=low|high;
*/

    unsigned char *plow, *phigh;
    plow = ( unsigned char * ) numberp;
    phigh = plow + 1;
    *numberp = *phigh * 256 + *plow;
    return;
}

/*********************************************/
void
Gif::unpack_init( short int clear )
{
    int i;
    for ( i = 0; i < clear; i++ )
    {
        CTfirst[i] = i;
        CTlast[i] = i;
        CTlink[i] = -1;
    }
    for ( i = clear; i < LISTLENGTH; i++ )
    {
        CTlink[i] = -2;
    }

    return;
}

/**********************************************/
void
Gif::insert_ct( short int code, short int oldcode )
{
    if ( nextcode >= LISTLENGTH )
        return;

    CTlink[nextcode] = oldcode;
    CTlast[nextcode] = CTfirst[code];
    CTfirst[nextcode++] = CTfirst[oldcode];
    if ( nextcode == nextlim && reqcnt < 12 )
    {
        reqcnt++;
        nextlim <<= 1;
    }

    return;
}

/**********************************************/
void
Gif::putx( short int code, char *data )
{
    int i;

    i = 0;
    while ( code != -1 && i < LISTLENGTH )
    {
        OStack[i++] = CTlast[code];
        code = CTlink[code];
    }

    int iLength = info.width * info.height;
    while ( i > 0 && datapoint < iLength )
    {
        data[datapoint++] = OStack[--i];
    }
    return;
}

/*******************************************************************/
short int
Gif::get_code( short int req, FILE * giffile )
{
    short int value, mask, start, used;

    start = 0;
    value = 0;
    while ( req > 0 )
    {
        if ( remcnt == 0 )
        {
            while ( index == 0 )
                index = getc( giffile );
            index--;
            rem = ( unsigned char ) getc( giffile );
            remcnt = 8;
        }
        if ( req > remcnt )
            used = remcnt;
        else
            used = req;
        remcnt -= used;
        mask = ( 0xff >> 8 - used );
        mask &= rem;
        rem >>= used;
        mask <<= start;
        start += used;
        value |= mask;
        req -= used;
    }
    return value;
}

/****************************************************************************/
Xpm::Xpm( char *filename )
{
    if ( filename != NULL && strlen( filename ) < LINELENGTH - 1 )
    {
        strcpy( xpmname, filename );
    } else
    {
        xpmname[0] = '\0';
    }
    return;
}

/****************************************************************************/
void
Xpm::vSetFileName( char *filename )
{
    if ( filename != NULL && strlen( filename ) < LINELENGTH - 1 )
    {
        strcpy( xpmname, filename );
    }

    return;
}

/****************************************************************************/
int
Xpm::iGetXpm( ImageInfo * xpminfo )
{
    char buff[10];
    int i;

    if ( !( xpmfile = fopen( xpmname, "rb" ) ) )
    {
        return ( errno );
    }

    for ( i = 0; i < 256; ++i )
    {
        info.palette[i].red = 0;
        info.palette[i].green = 0;
        info.palette[i].blue = 0;
    }
    info.colornum = 256;

    buff[0] = getc( xpmfile );
    buff[1] = getc( xpmfile );
    buff[2] = '\0';
    if ( !strcmp( buff, "/*" ) )
    {
        iFileType = MOTIFICON;
        vGetMotifIcon(  );
    } else if ( !strcmp( buff, "! " ) )
    {
        iFileType = OPENWINICON;
        vGetOpenwinIcon(  );
    } else
    {
        return ( NOTXPM );
    }

    fclose( xpmfile );
    *xpminfo = info;

    return ( 0 );
}

/*************************************************************/
void
Xpm::vGetMotifIcon(  )
{
    int i = 0, index, pixel, width, height;
    int colornum;
    int imagesize;

    seek( '"' );
    width = getnumber( 10 );
    info.width = width;
    height = getnumber( 10 );
    info.height = height;
    colornum = getnumber( 10 );

    seek( '"' );
    seek( '"' );

    info.bg_color = getc( xpmfile );

    seek( '"' );

    for ( i = 0; i < colornum - 1; ++i )
    {
        seek( '"' );
        index = getc( xpmfile );
        seek( '#' );
        pixel = getnumber( 16 );
        fillpalette( index, pixel );
        seek( ',' );
    }

    imagesize = info.width * info.height;
    info.data = ( char * ) malloc( imagesize );
    getdata(  );

    return;
}

/*************************************************************/
void
Xpm::vGetOpenwinIcon(  )
{
    int i = 0, index, pixel, width, height;
    int colornum;
    int imagesize;

    seek( '\n' );
    width = getnumber( 10 );
    info.width = width;
    height = getnumber( 10 );
    info.height = height;
    colornum = getnumber( 10 );

    seek( '\n' );

    info.bg_color = getc( xpmfile );

    seek( '\n' );

    for ( i = 0; i < colornum - 1; ++i )
    {
        index = getc( xpmfile );
        seek( '#' );
        pixel = getnumber( 16 );
        fillpalette( index, pixel );
    }

    imagesize = info.width * info.height;
    info.data = ( char * ) malloc( imagesize );
    getdata(  );

    return;
}

/*************************************************************/
int
Xpm::getnumber( int w )
{
    int ch, i;
    unsigned int data;
    char buff[10];

    if ( w == 16 )
    {
        while ( !isxdigit( ch = getc( xpmfile ) ) );    // jump over blanks.
        i = 0;
        do
        {
            buff[i++] = ch;
        }
        while ( isxdigit( ch = getc( xpmfile ) ) );

        if ( i > 6 )            // every color has 16 bits, change it to 8 bits.
        {
            buff[2] = buff[3];
            buff[3] = buff[5];
        }

        for ( i = 0, data = 0; i < 6; i++ )
        {
            data = data * 16 + char2number( buff[i] );
        }
    } else
    {
        while ( !isdigit( ch = getc( xpmfile ) ) );
        data = char2number( ch );
        while ( isdigit( ch = getc( xpmfile ) ) )
        {
            data = data * 10 + char2number( ch );
        }
    }

    return ( data );
}

/*************************************************************/
int
Xpm::char2number( int ch )
{
    if ( ch >= '0' && ch <= '9' )
    {
        return ( ch - '0' );
    } else if ( ch >= 'a' && ch <= 'f' )
    {
        return ( ch - 'a' + 10 );
    } else if ( ch >= 'A' && ch <= 'F' )
    {
        return ( ch - 'A' + 10 );
    } else
    {
        assert( 0 );
        return ( 0 );
    }
}

/*************************************************************/
void
Xpm::fillpalette( int index, int pixel )
{
    info.palette[index].blue = pixel & 0x000000ff;
    pixel >>= 8;
    info.palette[index].green = pixel & 0x000000ff;
    pixel >>= 8;
    info.palette[index].red = pixel & 0x000000ff;
    return;
}

/************************************************************/
void
Xpm::getdata(  )
{
    int i, j = 0;
    int ch;

    if ( iFileType == MOTIFICON )
    {
        for ( i = 0; i < info.height; ++i )
        {
            seek( '"' );

            while ( ( ch = getc( xpmfile ) ) != '"' )
            {
                info.data[j++] = ch;
            }
        }
    } else
    {
        for ( i = 0; i < info.height; ++i )
        {
            while ( ( ch = getc( xpmfile ) ) != '\n' )
            {
                info.data[j++] = ch;
            }
        }
    }

    return;
}

/***************************************************************************/
void
Xpm::seek( char ch )
{
    while ( getc( xpmfile ) != ch );
}

/********************************************************************/
Image::Image( char *filename )
{
    if ( filename != NULL && strlen( filename ) < LINELENGTH - 1 )
    {
        strcpy( sImageFile, filename );
    } else
    {
        sImageFile[0] = '\0';
    }
    return;
}

/********************************************************************/
void
Image::vSetFileName( char *filename )
{
    if ( filename != NULL && strlen( filename ) < LINELENGTH - 1 )
    {
        strcpy( sImageFile, filename );
    }

    return;
}

/********************************************************************/
void
Image::vSet( Screen * scrn, Colormap clrmap, Pixel bpixel )
{
    screen = scrn;
    colormap = clrmap;
    piBackgroundPixel = bpixel;
    return;
}

/********************************************************************/
inline void
Image::vDestroyImage(  )
{
    free( ximage->data );
    XFree( ximage );
    return;
}

/********************************************************************/
inline int
Image::iWidth(  )
{
    return ( imageinfo.width );
}

/********************************************************************/
inline int
Image::iHeight(  )
{
    return ( imageinfo.height );
}

/********************************************************************/
int
Image::iSearchBest( int iIndex )
{
    assert( iIndex < imageinfo.colornum && iIndex >= 0 );

    int iBest = 0;
    int iBestDiff = 256;

    RGBColor *pToMarch = imageinfo.palette + iIndex;
    int i;
    for ( i = 0; i < iIndex; ++i )
    {
        RGBColor *pNow = imageinfo.palette + i;

        int reddiff = abs( pNow->red - pToMarch->red );
        int greendiff = abs( pNow->green - pToMarch->green );
        int bluediff = abs( pNow->blue - pToMarch->blue );

        int maxdiff = reddiff;
        if ( maxdiff < greendiff )
            maxdiff = greendiff;
        if ( maxdiff < bluediff )
            maxdiff = bluediff;

        if ( maxdiff < iBestDiff )
        {
            iBest = i;
            iBestDiff = maxdiff;
        }
    }

    return ( iBest );
}

/********************************************************************/
int
Image::iGetImage( XImage * &image )
{
    int err, i;
    /* int red,green,blue; */
    /* XColor colors[256]; */
    XColor color;
    Pixel pixelchange[256];
    Display *display;
    Visual *visual;
    /*   int maxcolornum; */

    display = XDisplayOfScreen( screen );
    visual = XDefaultVisualOfScreen( screen );

    Gif oGif( sImageFile );
    Xpm oXpm( sImageFile );

    if ( !( err = oGif.iGetGif( &imageinfo ) ) )
    {
        iImageType = GIF;
    } else if ( !( err = oXpm.iGetXpm( &imageinfo ) ) )
    {
        iImageType = XPM;
    } else
    {
        return ( err );
    }

    int iDepth = DefaultDepth( display, DefaultScreen( display ) );
    ximage = XCreateImage( display, visual, iDepth,
                           ZPixmap, 0, NULL, imageinfo.width, imageinfo.height, 32, 0 );

    /* if colornum > maxcolornum ,we must ajust it by MOD */

/*
        maxcolornum=(256/COLORMOD+1)*(256/COLORMOD+1)*(256/COLORMOD+1);
        if(imageinfo.colornum>(256/COLORMOD+1))
        {
                for(i=0;i<imageinfo.colornum;++i)
                {
                        imageinfo.palette[i].red/=COLORMOD;
                        imageinfo.palette[i].red*=COLORMOD;
                        imageinfo.palette[i].red+=COLORMOD/2;
                        imageinfo.palette[i].green/=COLORMOD;
                        imageinfo.palette[i].green*=COLORMOD;
                        imageinfo.palette[i].green+=COLORMOD/2;
                        imageinfo.palette[i].blue/=COLORMOD;
                        imageinfo.palette[i].blue*=COLORMOD;
                        imageinfo.palette[i].blue+=COLORMOD/2;
                }
        }
*/

    iColorLost = 0;
    for ( i = 0; i < imageinfo.colornum; ++i )
    {
        color.red = imageinfo.palette[i].red << 8;
        color.green = imageinfo.palette[i].green << 8;
        color.blue = imageinfo.palette[i].blue << 8;
        if ( XAllocColor( display, colormap, &color ) )
        {
            pixelchange[i] = color.pixel;
        } else
        {
            pixelchange[i] = iSearchBest( i );
            iColorLost++;
        }
    }

    if ( iImageType == XPM )
    {
        pixelchange[imageinfo.bg_color] = piBackgroundPixel;
    }

    char *imagedata = ( char * ) malloc( ximage->bytes_per_line * imageinfo.height );
    ximage->data = imagedata;
    int x, y;
    for ( y = 0; y < imageinfo.height; ++y )
    {
        for ( x = 0; x < imageinfo.width; ++x )
        {
            int iIndex;
            Pixel realPixel;

            iIndex = imageinfo.width * y + x;
            realPixel = pixelchange[( unsigned ) imageinfo.data[iIndex]];
            XPutPixel( ximage, x, y, realPixel );
        }
    }
    free( imageinfo.data );

    image = ximage;

    return ( 0 );
}

/********************************************************************/
PixmapClass::PixmapClass( char *filename, Boolean isbitmap )
{
    if ( filename != NULL && strlen( filename ) < LINELENGTH - 1 )
    {
        strcpy( sImageFile, filename );
    } else
    {
        sImageFile[0] = '\0';
    }

    bIsBitmap = isbitmap;
    display = NULL;
    pixmap = XmUNSPECIFIED_PIXMAP;

    return;
}

/********************************************************************/
void
PixmapClass::vSetFileName( char *filename )
{
    if ( filename != NULL && strlen( filename ) < LINELENGTH - 1 )
    {
        strcpy( sImageFile, filename );
    }

    return;
}

/********************************************************************/
void
PixmapClass::vSet( Screen * scrn, Colormap clrmap, Pixel fpixel, Pixel bpixel )
{
    screen = scrn;
    colormap = clrmap;
    piBackgroundPixel = bpixel;
    piForegroundPixel = fpixel;
    return;
}

/********************************************************************/
int
PixmapClass::iGetWidth(  )
{
    return ( iWidth );
}

/********************************************************************/
int
PixmapClass::iGetHeight(  )
{
    return ( iHeight );
}

/********************************************************************/
Pixmap
PixmapClass::pixGetPixmap(  )
{
    return ( pixmap );
}

/********************************************************************/
PixmapClass::~PixmapClass(  )
{
/*
        if(display!=NULL && pixmap!=XmUNSPECIFIED_PIXMAP)
        {
                XFreePixmap(display,pixmap);
        }
*/
    return;
}

/********************************************************************/
void
PixmapClass::vCreate(  )
{
    int err;
    GC gc;
    unsigned int width, height;

    display = XDisplayOfScreen( screen );

    if ( bIsBitmap )
    {
        int status;

        status = XReadBitmapFile( display, XRootWindowOfScreen( screen ),
                                  sImageFile, &width, &height, &pixmap, &iX_Hot, &iY_Hot );

        if ( status != BitmapSuccess )
        {
            pixmap = XmUNSPECIFIED_PIXMAP;
            iWidth = 0;
            iHeight = 0;
        } else
        {
            iWidth = width;
            iHeight = height;
        }
    } else
    {
        Image oImage( sImageFile );
        oImage.vSet( screen, colormap, piBackgroundPixel );
        err = oImage.iGetImage( ximage );
        if ( !err )
        {
            int iDepth = DefaultDepth( display, DefaultScreen( display ) );
            pixmap = XCreatePixmap( display,
                                    XRootWindowOfScreen( screen ),
                                    ximage->width, ximage->height, iDepth );
            gc = XCreateGC( display, pixmap, 0, NULL );
            XPutImage( display, pixmap, gc, ximage, 0, 0, 0, 0, ximage->width, ximage->height );
//            XSync(display,False);
            XFreeGC( display, gc );
            oImage.vDestroyImage(  );
            iWidth = oImage.iWidth(  );
            iHeight = oImage.iHeight(  );
            iColorLost = oImage.iGetColorLost(  );
        } else
        {
            iWidth = 0;
            iHeight = 0;
            pixmap = XmUNSPECIFIED_PIXMAP;
        }
    }

    return;
}

/********************************** END *************************************/
